import {registerFormatType, applyFormat, removeFormat, unregisterFormatType} from '@wordpress/rich-text';
import {BlockControls, URLPopover} from '@wordpress/block-editor';
import {useState, useEffect} from '@wordpress/element';
import {TextControl, SelectControl, ToggleControl, Button, ToolbarButton, ToolbarGroup} from '@wordpress/components';
import {useSelect} from '@wordpress/data';
import domReady from '@wordpress/dom-ready';
import './styles.scss';

// === BEGIN CONFIGURATION ===
// List of allowed blocks where the custom link popover is enabled
const ALLOWED_BLOCKS = [
    'core/paragraph',
    'core/heading',
    'core/list',
    'core/list-item',
    'core/footnotes',
];
// Array of link skins with their selector label and class name (no default skin)
const LINK_SKINS = [
    { label: 'Link with arrow', className: 'arrow-link' },
    { label: 'Button Primary', className: 'btn-primary' },
    { label: 'Button White', className: 'btn-white' },
];
// === END CONFIGURATION ===

domReady(() => {
    // Unregister the default core/link format so we can replace it with our custom one
    unregisterFormatType('core/link');

    registerFormatType('core/link', {
        title: 'Link',
        tagName: 'a',
        className: null,
        attributes: {
            rel: 'rel',
            url: 'href',
            target: 'target',
            class: 'class',
        },

        edit({isActive, activeAttributes, value, onChange}) {
            // Only allow this format in allowed blocks
            const selectedBlock = useSelect((select) => {
                return select('core/block-editor').getSelectedBlock();
            }, []);

            if (
                selectedBlock &&
                !ALLOWED_BLOCKS.includes(selectedBlock.name)
            ) {
                return null;
            }

            const [isAddingLink, setIsAddingLink] = useState(false);
            const [url, setUrl] = useState(activeAttributes.url || '');

            // Determines the initial skin index based on the link's class attribute
            const getInitialSkinIndex = (classAttr = '') => {
                for (let i = 0; i < LINK_SKINS.length; i++) {
                    if (LINK_SKINS[i].className && classAttr.includes(LINK_SKINS[i].className)) {
                        return i;
                    }
                }
                return -1; // No skin by default
            };

            const [skinIndex, setSkinIndex] = useState(getInitialSkinIndex(activeAttributes.class || ''));

            // Advanced options toggles
            const [nofollow, setNofollow] = useState(activeAttributes.rel?.includes('nofollow') || false);
            const [openInNewTab, setOpenInNewTab] = useState(activeAttributes.target === '_blank' || false);
            const [relNoRefNoOp, setRelNoRefNoOp] = useState(activeAttributes.rel?.includes('noopener noreferrer') || false);

            // Control for showing/hiding advanced options
            const [showAdvancedOptions, setShowAdvancedOptions] = useState(false);

            useEffect(() => {
                if (isActive) {
                    setUrl(activeAttributes.url || '');
                    setNofollow(activeAttributes.rel?.includes('nofollow') || false);
                    setOpenInNewTab(activeAttributes.target === '_blank' || false);
                    setRelNoRefNoOp(activeAttributes.rel?.includes('noopener noreferrer') || false);

                    setSkinIndex(getInitialSkinIndex(activeAttributes.class || ''));
                }
            }, [isActive, activeAttributes]);

            /**
             * Toggles the display of the URL popover panel.
             */
            const togglePopover = () => {
                if (!isAddingLink) {
                    setUrl(activeAttributes.url || '');
                    setNofollow(activeAttributes.rel?.includes('nofollow') || false);
                    setOpenInNewTab(activeAttributes.target === '_blank' || false);
                    setRelNoRefNoOp(activeAttributes.rel?.includes('noopener noreferrer') || false);

                    setSkinIndex(getInitialSkinIndex(activeAttributes.class || ''));
                }
                setIsAddingLink(!isAddingLink);
            };

            /**
             * Closes the popover without applying changes.
             */
            const closePopover = () => {
                setIsAddingLink(false);
            };

            /**
             * Builds the class string for the link based on the selected skin.
             * Returns an empty string if no skin is selected.
             */
            const buildClassName = () => {
                if (skinIndex === -1) return '';
                return LINK_SKINS[skinIndex].className;
            };

            /**
             * Applies the link format with updated attributes to all links inside allowed blocks.
             * Updates attributes for any <a> tag, regardless of its class.
             */
            const addLink = () => {
                if (typeof url !== 'string') {
                    return;
                }

                let newAttributes = {
                    url: url,
                    rel: (nofollow ? 'nofollow ' : '') + (relNoRefNoOp ? 'noopener noreferrer' : ''),
                    target: openInNewTab ? '_blank' : '_self',
                };
                // Only add class attribute if a skin is selected
                const className = buildClassName();
                if (className) {
                    newAttributes.class = className;
                } else {
                    newAttributes.class = '';
                }

                try {
                    onChange(
                        applyFormat(value, {
                            type: 'core/link',
                            attributes: newAttributes,
                        })
                    );
                } catch (e) {
                    console.error('Error applying format:', e);
                }

                setIsAddingLink(false);
            };

            /**
             * Removes the link format completely.
             */
            const removeLink = () => {
                onChange(removeFormat(value, 'core/link'));
                setIsAddingLink(false);
            };

            /**
             * Toggles the visibility of the advanced options panel.
             */
            const toggleAdvancedOptions = () => {
                setShowAdvancedOptions(!showAdvancedOptions);
            };

            return (
                <>
                    <BlockControls>
                        <ToolbarGroup>
                            <ToolbarButton
                                icon="admin-links"
                                title="Link Options"
                                isActive={isActive}
                                onClick={togglePopover}
                                className="onev-link-button"
                            />
                        </ToolbarGroup>
                    </BlockControls>

                    {isAddingLink && (
                        <URLPopover onClose={closePopover} className="onev-link-popover">
                            <div className="onev-link-controls">
                                <TextControl
                                    label="URL"
                                    value={url}
                                    onChange={(newUrl) => setUrl(newUrl)}
                                />

                                <SelectControl
                                    label="Skin"
                                    value={skinIndex}
                                    options={LINK_SKINS.map((skin, idx) => ({ label: skin.label, value: idx })).concat([{ label: 'No skin', value: -1 }])}
                                    onChange={(newIdx) => setSkinIndex(Number(newIdx))}
                                />

                                <Button
                                    onClick={toggleAdvancedOptions}
                                    aria-expanded={showAdvancedOptions}
                                    aria-controls="advanced-options"
                                    className="advanced-options-button"
                                >
                                    Advanced options {showAdvancedOptions ? '▲' : '▼'}
                                </Button>

                                <div
                                    id="advanced-options"
                                    className={`advanced-options ${showAdvancedOptions ? 'expanded' : 'collapsed'}`}
                                >
                                    <ToggleControl
                                        label='Add "nofollow"'
                                        checked={nofollow}
                                        onChange={(newNofollow) => setNofollow(newNofollow)}
                                    />
                                    <ToggleControl
                                        label="Open in new tab"
                                        checked={openInNewTab}
                                        onChange={(newOpenInNewTab) => setOpenInNewTab(newOpenInNewTab)}
                                    />
                                    <ToggleControl
                                        label='Add "noopener noreferrer"'
                                        checked={relNoRefNoOp}
                                        onChange={(newRelNoRefNoOp) => setRelNoRefNoOp(newRelNoRefNoOp)}
                                    />
                                </div>

                                <div className="onev-link-buttons">
                                    <Button isPrimary onClick={addLink}>
                                        Update Link
                                    </Button>
                                    <Button isDestructive onClick={removeLink} className="onev-link-remove">
                                        Remove Link
                                    </Button>
                                </div>
                            </div>
                        </URLPopover>
                    )}
                </>
            );
        },
    });
});

// === COMMENTS ===
// ALLOWED_BLOCKS — list of blocks where the popover is enabled
// LINK_SKINS — array of skin objects with label and className, easy to extend
// getInitialSkinIndex — detects skin by class attribute, returns -1 for no skin
// buildClassName — builds class string for the link
// addLink — applies format to any <a> tag, regardless of its class
