import domReady from '@wordpress/dom-ready';

// Define constants for namespace and preview image filename
const BLOCK_NAMESPACE = 'acf-block';
const PREVIEW_IMAGE_FILENAME = 'preview.webp';

// Cache to store image existence check results and pending promises
const imageCache = new Map();
const pendingPromises = new Map();

// Debounce utility to limit handler execution
function debounce(func, wait) {
    let timeout;
    return function (...args) {
        clearTimeout(timeout);
        timeout = setTimeout(() => func.apply(this, args), wait);
    };
}

domReady(() => {
    // Debounced handler to prevent rapid repeated calls
    const handleMouseOver = debounce((e) => {
        // Get the preview container element
        const previewContainer = document.querySelector('.block-editor-inserter__preview-content-missing');
        if (!previewContainer) {
            return;
        }

        // Check if the mouse is over a block item
        const hoveredBlock = e.target.closest('.block-editor-block-types-list__item');
        if (!hoveredBlock) {
            return;
        }

        // Extract block classes and find the one matching the namespace
        const blockClasses = hoveredBlock.className.split(' ');
        const blockClassPrefix = `editor-block-list-item-${BLOCK_NAMESPACE}-`;
        const blockClass = blockClasses.find(cls => cls.startsWith(blockClassPrefix));
        if (!blockClass) {
            return;
        }

        // Extract the block name by removing the prefix
        const blockName = blockClass.replace(blockClassPrefix, "");
        const blockData = wp.data.select('core/blocks').getBlockType(`${BLOCK_NAMESPACE}/${blockName}`);

        // Construct image URL from a block path
        let imageUrl = '';
        if (blockData.path) {
            const wpContentIndex = blockData.path.indexOf('wp-content');
            if (wpContentIndex !== -1) {
                const relativePath = blockData.path.substring(wpContentIndex);
                imageUrl = `${window.location.origin}/${relativePath}/${PREVIEW_IMAGE_FILENAME}`;
            }
        }

        if (imageUrl) {
            // Check cache for an existing result
            if (imageCache.has(imageUrl)) {
                const isImageValid = imageCache.get(imageUrl);
                if (isImageValid) {
                    applyImageStyles(previewContainer, imageUrl);
                } else {
                    resetImageStyles(previewContainer);
                }
                return;
            }

            // Check if a fetch is already pending for this URL
            if (pendingPromises.has(imageUrl)) {
                // Await the existing promise
                pendingPromises.get(imageUrl).then(() => {
                    const isImageValid = imageCache.get(imageUrl);
                    if (isImageValid) {
                        applyImageStyles(previewContainer, imageUrl);
                    } else {
                        resetImageStyles(previewContainer);
                    }
                });
                return;
            }

            // Start a new fetch and store the promise
            const fetchPromise = fetch(imageUrl, { method: 'HEAD' })
                .then(response => {
                    const isImageValid = response.ok;
                    imageCache.set(imageUrl, isImageValid); // Store result in cache
                    pendingPromises.delete(imageUrl); // Clear pending promise
                    if (isImageValid) {
                        applyImageStyles(previewContainer, imageUrl);
                    } else {
                        resetImageStyles(previewContainer);
                    }
                })
                .catch(() => {
                    imageCache.set(imageUrl, false); // Cache as invalid on error
                    pendingPromises.delete(imageUrl); // Clear pending promise
                    resetImageStyles(previewContainer);
                });

            pendingPromises.set(imageUrl, fetchPromise);
        } else {
            resetImageStyles(previewContainer);
        }
    }, 100); // Debounce with 100ms delay

    // Attach debounced handler to mouseover event
    document.addEventListener('mouseover', handleMouseOver);

    // Apply image styles to the preview container
    function applyImageStyles(container, imageUrl) {
        container.style.background = `url(${imageUrl}) no-repeat center`;
        container.style.backgroundSize = 'contain';
        container.style.fontSize = '0px';
    }

    // Reset preview container styles
    function resetImageStyles(container) {
        container.style.background = '';
        container.style.backgroundSize = '';
        container.style.fontSize = '';
    }
});