<?php

declare( strict_types=1 );

namespace Headless;

use Exception;
use Headless\Kernel\Core;
use Headless\Interfaces\Registrable;
use ReflectionClass;
use ReflectionException;

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Bootstraps modules based on a configuration file
 *
 * This function reads the list of module class names from config.php and triggers
 * their registration via the Loadable trait. It relies on Composer's PSR-4 autoloading
 * to load the module files when their classes are referenced
 *
 * @throws Exception
 */
function bootstrap_modules(): void {

	// Load the configuration file containing the list of modules
	$config = require __DIR__ . '/config.php';

	// Check if the 'modules' key exists in the config
	if ( ! isset( $config['modules'] ) || ! is_array( $config['modules'] ) ) {
		error_log( 'Error: No valid "modules" array found in config.php' );

		return;
	}

	// Define constants from config
	if ( isset( $config['constants'] ) && is_array( $config['constants'] ) ) {
		foreach ( $config['constants'] as $name => $value ) {
			if ( ! defined( $name ) ) {
				define( $name, $value );
			}
		}
	}

	// Iterate over each module class name specified in the config
	array_walk( $config['modules'], function ( string $class_name ) {
		try {
			// Ensure the class exists
			if ( class_exists( $class_name ) ) {

				// Use ReflectionClass to inspect the class properties
				$reflection = new ReflectionClass( $class_name );

				// Verify that the class implements Registrable and is not abstract
				if ( $reflection->implementsInterface( Registrable::class ) && ! $reflection->isAbstract() ) {

					// Call the static load() method provided by the Loadable trait
					// This registers the class with Core
					$class_name::load();
				}
			} else {
				error_log( "Class $class_name does not exist or failed to autoload" );
			}
		} catch ( ReflectionException $e ) {
			error_log( "Reflection error for class $class_name: " . $e->getMessage() );
		}
	} );

	// Initialize the Core class
	Core::get_instance()->init();

}

// Execute the bootstrapping process
try {
	bootstrap_modules();
} catch ( Exception $e ) {
	error_log( $e->getMessage() );
}