<?php

declare( strict_types=1 );

namespace Headless\Utils;

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Class CommonUtils
 *
 * Utility class for usage across the theme
 */
final class CommonUtils {

	/**
	 * Get form from CPT by post id
	 *
	 * @param int $post_id Post id
	 *
	 * @return array|null Form data
	 */
	public static function get_form( int $post_id ): ?array {
		$fields = get_fields( $post_id );

		if ( empty( $fields ) ) {
			return null;
		}

		return $fields;
	}

	/**
	 * Render a placeholder for the admin editor.
	 *
	 * @param string $title The title of the placeholder.
	 * @param string $instructions The instructions for the placeholder.
	 *
	 * @return string The HTML for the placeholder.
	 */
	public static function admin_placeholder( string $title, string $instructions ): string {

		if ( is_admin() ) {
			return '<div class="components-placeholder block-editor-media-placeholder is-large">
            <div class="components-placeholder__label">
                ' . esc_html( $title ) . '
            </div>
            <div class="components-placeholder__instructions">' . esc_html( $instructions ) . '</div>
        </div>';
		}

		return '';
	}

	/**
	 * Filters a link properties array
	 *
	 * @param mixed $link_props The link properties array
	 * @param array|null $classes Optional array of CSS classes to add to the image
	 *
	 * @return array|null The filtered link properties array
	 */
	public static function prepare_link_props( mixed $link_props, ?array $classes = null ): ?array {

		$url = $link_props['href'] ?? $link_props['url'] ?? '#';

		return [
			'url'     => esc_url( $url ),
			'target'  => esc_attr( $link_props['target'] ?? '' ),
			'title'   => esc_html( $link_props['title'] ?? '' ),
			'classes' => is_array( $classes ) ? $classes : [],
		];
	}

	/**
	 * Filters an image properties array
	 *
	 * @param mixed $image_props The image properties array
	 * @param array|null $classes Optional array of CSS classes to add to the image
	 * @param string|null $size Optional size of the placeholer image
	 *
	 * @return array|null The filtered image properties array
	 */
	public static function prepare_image_props( array|false $image_props, ?array $classes = null, ?string $size = null ): ?array {

		$size = $size ?? '600x400';

		$src = $image_props['src'] ?? $image_props['url'] ?? 'https://placehold.co/' . $size . '.png';

		return [
			'src'     => esc_url( $src ),
			'width'   => intval( $image_props['width'] ?? 0 ),
			'height'  => intval( $image_props['height'] ?? 0 ),
			'alt'     => esc_attr( $image_props['alt'] ?? '' ),
			'classes' => is_array( $classes ) ? $classes : [],
		];
	}

	/**
	 * Generates an array of image properties for a given image ID
	 *
	 * @param $image_id int The ID of the image
	 *
	 * @return array|null Image properties including src, width, height, and alt text
	 */
	public static function get_image_by_id( int $image_id ): ?array {

		$image = wp_get_attachment_image_src( $image_id, 'full' );

		// Check if the image exists
		if ( ! $image ) {
			return null;
		}

		$alt = get_post_meta( $image_id, '_wp_attachment_image_alt', true );

		return [
			'src'    => esc_url( $image[0] ),
			'width'  => intval( $image[1] ),
			'height' => intval( $image[2] ),
			'alt'    => esc_attr( $alt ),
		];
	}

	/**
	 * Generates the HTML for a given image properties array
	 *
	 * @param array|null $image_props The image properties array
	 *
	 * @return string|null The HTML for the image
	 */
	public static function render_image( ?array $image_props ): ?string {

		if ( ! $image_props ) {
			return null;
		}

		// Extract properties
		$src     = $image_props['src'] ?? '';
		$alt     = $image_props['alt'] ?? '';
		$classes = $image_props['classes'] ?? [];
		$width   = $image_props['width'] ?? '';
		$height  = $image_props['height'] ?? '';

		// Prepare attributes
		$alt_attr = ! empty( $alt ) ? ' alt="' . esc_attr( $alt ) . '"' : '';

		$class_attr = '';
		if ( is_array( $classes ) && ! empty( $classes ) ) {
			$sanitized_classes = array_map( 'sanitize_html_class', $classes );
			$class_attr        = ' class="' . esc_attr( implode( ' ', $sanitized_classes ) ) . '"';
		}

		$width_attr  = is_numeric( $width ) && $width > 0 ? ' width="' . intval( $width ) . '"' : '';
		$height_attr = is_numeric( $height ) && $height > 0 ? ' height="' . intval( $height ) . '"' : '';

		// Build the HTML using a template string approach
		return sprintf(
			'<img src="%s"%s%s%s%s />',
			esc_url( $src ),
			$alt_attr,
			$class_attr,
			$width_attr,
			$height_attr
		);
	}
	
	/**
	 * Renders an ACF image field as an <img> tag.
	 *
	 * @param array|null $image ACF image array.
	 * @param array       $args Optional. Additional arguments like classes, size. Default empty array.
	 *
	 * @return string|null HTML output or null if invalid input.
	 */
	public static function render_image_acf( $image, array $args = [] ): ?string {
		if ( empty( $image ) ) return null;
	
		// === Defaults
		$defaults = [
			'width'         => null,
			'height'        => null,
			'alt'           => '',
			'class'         => [],
			'loading'       => 'lazy',
			'decoding'      => 'async',
			'fetchpriority' => 'auto',
			'size'          => null,   // which WP size to use for src/srcset calculation
			'sizes'         => null,   // sizes attribute; auto-computed if empty
		];
		$args = array_merge( $defaults, $args );
	
		// === Resolve attachment id and URL
		$attachment_id = null;
		$url           = null;
	
		if ( is_array( $image ) ) {
			$attachment_id = intval( $image['ID'] ?? $image['id'] ?? 0 );
			$url           = $image['url'] ?? null;
			if ( ! $attachment_id && $url ) {
				$attachment_id = attachment_url_to_postid( $url );
			}
		} elseif ( is_numeric( $image ) ) {
			$attachment_id = (int) $image;
		}
	
		if ( ! $attachment_id ) return null;
	
		if ( ! $url ) {
			$src = wp_get_attachment_image_src( $attachment_id, 'full' );
			$url = is_array( $src ) ? $src[0] : wp_get_attachment_url( $attachment_id );
		}
		if ( ! $url ) return null;
	
		// === Basic info
		$mime  = get_post_mime_type( $attachment_id ) ?: '';
		$is_svg = stripos( $mime, 'image/svg' ) === 0 || preg_match( '/\.svg(\?.*)?$/i', $url );
	
		$meta   = wp_get_attachment_metadata( $attachment_id );
		$orig_w = is_array( $meta ) ? (int) ( $meta['width']  ?? 0 ) : 0;
		$orig_h = is_array( $meta ) ? (int) ( $meta['height'] ?? 0 ) : 0;
	
		$w_attr = $args['width']  ?: ( $orig_w ?: null );
		$h_attr = $args['height'] ?: ( $orig_h ?: null );
	
		// === Alt text
		$alt = trim( (string) $args['alt'] );
		if ( $alt === '' ) {
			$media_alt = get_post_meta( $attachment_id, '_wp_attachment_image_alt', true );
			$alt = $media_alt ?: '';
		}
	
		// === Classes
		$class = $args['class'];
		if ( ! is_array( $class ) ) {
			$class = preg_split( '/\s+/', (string) $class, -1, PREG_SPLIT_NO_EMPTY );
		}
		$class_str  = implode( ' ', array_map( 'sanitize_html_class', $class ) );
		$class_attr = $class_str ? ' class="' . esc_attr( $class_str ) . '"' : '';
	
		// === Heuristic: tiny icons/SVGs do not need heavy srcset logic
		$likely_icon = ( $w_attr && $w_attr <= 80 ) && ( $h_attr && $h_attr <= 80 );
	
		if ( $is_svg || $likely_icon ) {
			$alt_attr           = ' alt="' . esc_attr( $alt ) . '"'; // empty alt is acceptable for decorative
			$loading_attr       = ' loading="' . esc_attr( $args['loading'] ) . '"';
			$decoding_attr      = ' decoding="' . esc_attr( $args['decoding'] ) . '"';
			$fetchpriority_attr = ' fetchpriority="' . esc_attr( $args['fetchpriority'] ) . '"';
			$width_attr         = $w_attr ? ' width="' . (int) $w_attr . '"' : '';
			$height_attr        = $h_attr ? ' height="' . (int) $h_attr . '"' : '';
	
			return sprintf(
				'<img src="%s"%s%s%s%s%s%s />',
				esc_url( $url ),
				$alt_attr,
				$class_attr,
				$width_attr,
				$height_attr,
				$loading_attr,
				$decoding_attr . $fetchpriority_attr
			);
		}
	
		// === Choose a WP size to compute src/srcset
		$chosen_size = $args['size'];
		if ( $chosen_size === null ) {
			// If original is not huge, prefer 'full', else 'large' to save bandwidth
			$chosen_size = ( $orig_w && $orig_w <= 1000 ) ? 'full' : 'large';
		}
	
		// === Compute sizes attribute if not provided
		$computed_sizes = $args['sizes'];
		if ( ! $computed_sizes ) {
			// desktop width fallback: either desired width or capped original
			$desktop_px     = $w_attr ?: ( $orig_w ? min( $orig_w, 1000 ) : 996 );
			$computed_sizes = "(max-width: 768px) 100vw, (max-width: 1200px) 90vw, {$desktop_px}px";
		}
	
		// === Common attributes
		$loading_attr       = ' loading="' . esc_attr( $args['loading'] ) . '"';
		$decoding_attr      = ' decoding="' . esc_attr( $args['decoding'] ) . '"';
		$fetchpriority_attr = ' fetchpriority="' . esc_attr( $args['fetchpriority'] ) . '"';
		$alt_attr           = ' alt="' . esc_attr( $alt ) . '"';
	
		// === If explicit width/height are provided, build the <img> manually
		// to prevent wp_get_attachment_image() from overriding them.
		if ( $w_attr || $h_attr ) {
			$src    = wp_get_attachment_image_url( $attachment_id, $chosen_size ) ?: $url;
			$srcset = wp_get_attachment_image_srcset( $attachment_id, $chosen_size );
	
			$parts = [];
			$parts[] = 'src="' . esc_url( $src ) . '"';
			if ( $srcset ) {
				$parts[] = 'srcset="' . esc_attr( $srcset ) . '"';
				$parts[] = 'sizes="' . esc_attr( $computed_sizes ) . '"';
			}
			$parts[] = substr( $alt_attr, 1, -0 ); // keep as prepared
			if ( $class_str ) {
				$parts[] = 'class="' . esc_attr( $class_str ) . '"';
			}
			if ( $w_attr ) $parts[] = 'width="' . (int) $w_attr . '"';
			if ( $h_attr ) $parts[] = 'height="' . (int) $h_attr . '"';
			$parts[] = substr( $loading_attr, 1, -0 );
			$parts[] = substr( $decoding_attr, 1, -0 );
			$parts[] = substr( $fetchpriority_attr, 1, -0 );
	
			return '<img ' . implode( ' ', $parts ) . ' />';
		}
	
		// === Otherwise we can safely use the WP helper (it will set width/height from the size)
		$attr = [
			'class'         => $class_str,
			'alt'           => $alt,
			'loading'       => $args['loading'],
			'decoding'      => $args['decoding'],
			'fetchpriority' => $args['fetchpriority'],
			'sizes'         => $computed_sizes,
		];
		return wp_get_attachment_image( $attachment_id, $chosen_size, false, $attr );
	}
	

    /**
     * Returns inline styles for background images (desktop and mobile) from ACF image arrays or URLs.
     *
     * @param array|string|null $desktop_image ACF image array or URL for desktop background.
     * @param array|string|null $mobile_image  Optional. ACF image array or URL for mobile background.
     * @param string|null        $size         Optional. Image size to use for ACF arrays. Default 'large'.
     *
     * @return string|null Inline style attribute or null if no valid desktop image.
     */
    public static function render_bg_style_responsive( array|string|null $desktop_image, array|string|null $mobile_image = null, ?string $size = 'large' ): ?string {
        // Normalize desktop image
        if ( is_array( $desktop_image ) ) {
            $desktop_src = $desktop_image['sizes'][ $size ] ?? $desktop_image['url'] ?? null;
        } elseif ( is_string( $desktop_image ) ) {
            $desktop_src = $desktop_image;
        } else {
            $desktop_src = null;
        }

        if ( empty( $desktop_src ) ) {
            return null;
        }

        // Normalize mobile image
        if ( is_array( $mobile_image ) ) {
            $mobile_src = $mobile_image['sizes'][ $size ] ?? $mobile_image['url'] ?? '';
        } elseif ( is_string( $mobile_image ) ) {
            $mobile_src = $mobile_image;
        } else {
            $mobile_src = '';
        }

        // Build style attribute
        $style = 'style="--bg-desktop: url(' . esc_url( $desktop_src ) . ');';

        if ( ! empty( $mobile_src ) ) {
            $style .= '--bg-mobile: url(' . esc_url( $mobile_src ) . ');';
        }

        $style .= '"';

        return $style;
    }

    /**
	 * Generates the HTML for a link based on ACF link field data
	 *
	 * @param array|null $link_props The link properties array from ACF link field
	 *
	 * @return string|null The HTML for the link
	 */
	public static function render_link( ?array $link_props ): ?string {
		if ( ! $link_props ) {
			return null;
		}

		// Default URL to # if empty
		$url     = ! empty( $link_props['url'] ) ? $link_props['url'] : '#';
		$target  = $link_props['target'] ?? '';
		$title   = $link_props['title'] ?? '';
		$classes = $link_props['classes'] ?? [];

		// Prepare class attribute
		$class_attr = '';
		if ( is_array( $classes ) && ! empty( $classes ) ) {
			$sanitized_classes = array_map( 'sanitize_html_class', $classes );
			$class_attr        = ' class="' . esc_attr( implode( ' ', $sanitized_classes ) ) . '"';
		}

		// Prepare target and rel attributes
		$target_attr = '';
		$rel_attr    = '';
		if ( ! empty( $target ) ) {
			$target_attr = ' target="' . esc_attr( $target ) . '"';

			if ( $target === '_blank' ) {
				$rel_attr = ' rel="noopener noreferrer"';
			}
		}

		// Link text content
		$link_text = ! empty( $title ) ? $title : $url;

		// Build the HTML using a template string approach
		return sprintf(
			'<a tabindex="0" href="%s"%s%s%s>%s</a>',
			esc_url( $url ),
			$target_attr,
			$rel_attr,
			$class_attr,
			esc_html( $link_text )
		);
	}

	/**
	 * Converts a snake_case string to camelCase
	 *
	 * @param string $string The snake_case string to convert
	 *
	 * @return string The camelCase string
	 */
	public static function snake_to_camel( string $string ): string {

		// Convert the string to lowercase
		$string = strtolower( $string );

		// Replace spaces with underscores
		$string = str_replace( ' ', '_', $string );

		// Split the string into parts
		$parts = explode( '_', $string );

		// Initialize the camelCased string
		$camel_cased = strtolower( array_shift( $parts ) );

		// Iterate over the parts and uppercase the first letter of each part
		foreach ( $parts as $part ) {
			$camel_cased .= ucfirst( $part );
		}

		// Return the camelCased string
		return $camel_cased;
	}
}