<?php

declare( strict_types=1 );

namespace Headless\Utils\Block;

use Headless\Utils\CommonUtils;
use WP_Post;

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Class BlogArticleUtils
 *
 * Utility class for handling blog article data
 */
final class BlogArticleUtils {

	/**
	 * Get categories data array from an array of category IDs
	 *
	 * @param array|null $categories Array of category IDs
	 *
	 * @return array|null Categories data array
	 */
	public static function get_categories_array( ?array $categories ): ?array {

		if ( ! $categories ) {
			// Return empty string if the categories are not set
			return null;
		}

		// Initialize the categories array
		$categories_data = [];

		// Get the categories from the passed IDs
		return array_map(
			static function ( $category_id ) {

				// Get the term for the ID
				$term = get_term_by( 'id', (int) $category_id, 'blog_category' );

				// Return empty string if the term does not exist
				if ( ! $term ) {
					return null;
				}

				// Return the category
				return [
					'name' => esc_html( $term->name ),
					'url'  => esc_attr( '?current-category=' . intval( $term->term_id ) )
				];
			},
			$categories
		);
	}


	/**
	 * Get categories HTML from an array of category IDs
	 *
	 * @param array|null $categories Array of category IDs
	 *
	 * @return string|null Categories HTML
	 */
	public static function get_categories_html( ?array $categories ): ?string {

		// Return empty string if the categories are not set
		if ( ! $categories ) {
			return '';
		}

		// Initialize the categories HTML
		$html = '';

		// Loop through the categories from the passed IDs
		foreach ( $categories as $category ) {

			// Create the link for the category
			if ( ! empty( $category['url'] ) && ! empty( $category['name'] ) ) {
				$html .= '<a href="' . esc_url( $category['url'] ) . '">' . esc_html( $category['name'] ) . '</a>';
			}

		}

		return $html;
	}

	/**
	 * Get author data array from an author ID
	 *
	 * @param int|null $author_id Author ID
	 *
	 * @return array|null Author data array
	 */
	public static function get_author_array( ?int $author_id ): ?array {

		if ( ! $author_id ) {
			// Return empty string if the author ID is not set
			return null;
		}

		// Get the term for the author ID
		$term = get_term_by( 'id', (int) $author_id, 'blog_author' );
		if ( ! $term ) {
			// Return empty string if the term does not exist
			return null;
		}

		// Initialize the author data array
		$author_data = [];

		// Get the name of the author
		$author_data['name'] = esc_html( $term->name );

		// Create the URL for the author
		$author_data['url'] = esc_attr( '/blog/author/' . esc_attr( $term->slug ) );

		// Get the position of the author
		$position                = get_field( 'position', 'blog_author_' . $term->term_id );
		$author_data['position'] = ! empty( $position ) ? esc_html( $position ) : null;

		return $author_data;
	}


	/**
	 * Get author HTML from an author data array
	 *
	 * @param array|null $author_array Author data array
	 *
	 * @return string Author HTML
	 */
	public static function get_author_html( ?array $author_array ): string {

		if ( ! $author_array ) {
			// Return empty string if the author ID is not set
			return '';
		}

		// Create the HTML for the author
		$html = "<a href=\"{$author_array['url']}\">{$author_array['name']}</a>";

		// If the author has a position, add it to the HTML
		if ( ! empty( $author_array['position'] ) ) {
			$html .= ", {$author_array['position']}";
		}

		return $html;
	}

	/**
	 * Get formatted related posts
	 *
	 * @param array $fields The ACF fields
	 * @param int $post_id The ID of the post for which to get related posts
	 *
	 * @return array|null Formatted related posts
	 */
	public static function get_related_posts_array( array $fields, int $post_id ): ?array {

		// If the related type is custom and related posts are set, return the formatted related posts
		if ( ! empty( $fields['related_type'] ) && $fields['related_type'] === 'custom' ) {
			return self::get_custom_related_posts_array( $fields, $post_id );
		} else {
			return self::get_default_related_posts_array( $fields, $post_id );
		}
	}


	/**
	 * Get an array of formatted related posts if the related type is custom and related posts are set
	 *
	 * @param array $fields The ACF fields
	 * @param int $post_id The ID of the post for which to get related posts
	 *
	 * @return array Formatted related posts
	 */
	private static function get_custom_related_posts_array( array $fields, int $post_id ): array {

		// Initialize the related posts array
		$related_posts = [];

		// If the related type is custom and related posts are set, loop through the related posts
		if ( ! empty( $fields['related_posts'] ) && is_array( $fields['related_posts'] ) ) {

			// Loop through the related posts
			foreach ( $fields['related_posts'] as $post ) {

				// Get the related post array
				$related_posts[] = self::get_related_post_array( $post );
			}
		}

		// Return the related posts array
		return $related_posts;
	}

	/**
	 * Get an array of formatted related posts
	 *
	 * @param array $fields The ACF fields
	 * @param int $post_id The ID of the post for which to get related posts
	 *
	 * @return array Formatted related posts
	 */
	private static function get_default_related_posts_array( array $fields, int $post_id ): array {

		// Get the total number of related posts to fetch
		// If the number of related posts is not set in the ACF fields, default to 3
		$total = ! empty( $fields['related_number'] ) ? (int) $fields['related_number'] : 3;

		// Get the categories for the post
		$categories = ! empty( $fields['blog_categories'] ) ? $fields['blog_categories'] : [];

		// Set up the query arguments
		$query_args = [
			'post_type'   => 'post',
			'post_status' => 'publish',
			'numberposts' => $total,
			'exclude'     => $post_id,
			'tax_query'   => [],
		];

		// If categories are set, add them to the tax query
		if ( ! empty( $categories ) ) {
			// Add the categories to the tax query
			$query_args['tax_query'][] = [
				'taxonomy' => 'blog_category',
				'field'    => 'id',
				'operator' => 'IN',
				'terms'    => array_map( 'intval', $categories ),
			];
		}

		// Get the related posts
		$posts = get_posts( $query_args );

		// Initialize the related posts array
		$related_posts = [];

		if ( $posts ) {
			// Loop through the related posts
			foreach ( $posts as $post ) {
				// Get the related post array
				$related_posts[] = self::get_related_post_array( $post );
			}
		}

		// Return the related posts array
		return $related_posts;
	}

	/**
	 * Return an array with a post's related data
	 *
	 * @param WP_Post $post The post object
	 *
	 * @return array The post's related data
	 */
	public static function get_related_post_array( WP_Post $post ): array {

		// Get the author for the post
		$author_terms = get_the_terms( $post->ID, 'blog_author' );
		$author_id    = ! empty( $author_terms ) && ! is_wp_error( $author_terms ) ? $author_terms[0]->term_id : null;
		$author_array = $author_id ? self::get_author_array( $author_id ) : null;

		// Get the categories for the post
		$categories_terms = get_the_terms( $post->ID, 'blog_category' );
		$categories_ids   = ! empty( $categories_terms ) && ! is_wp_error( $categories_terms ) ? wp_list_pluck( $categories_terms, 'term_id' ) : null;
		$categories_array = $categories_ids ? self::get_categories_array( $categories_ids ) : null;

		// Image
		// $image = CommonUtils::prepare_image_props( get_the_post_thumbnail( $post->ID ) );

		return [
			'title'      => $post->post_title,
			'link'       => get_permalink( $post->ID ),
			// 'image'      => $image,
			'date'       => self::format_date( $post->post_date ),
			'author'     => $author_array,
			'categories' => $categories_array,
		];
	}

	/**
	 * Format a timestamp into a human-readable date string
	 *
	 * @param string $timestamp The timestamp to format
	 *
	 * @return string The formatted date
	 */
	private static function format_date( string $timestamp = '' ): string {
		$t      = $timestamp ? strtotime( $timestamp ) : current_time( 'timestamp' );
		$day    = (int) date( 'j', $t );
		$suffix = match ( $day ) {
			1, 21, 31 => 'st',
			2, 22 => 'nd',
			3, 23 => 'rd',
			default => 'th',
		};

		return date_i18n( "F j", $t ) . $suffix . date_i18n( ", Y", $t );
	}
}