<?php

declare( strict_types=1 );

namespace Headless\Modules\Blocks;

use WP_Block_Editor_Context;

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Class BlockRestrict
 *
 * Handles the restriction of Gutenberg blocks based on post type and user role
 */

class BlockRestrict {
	/**
	 * Enqueue view assets for blog article block on single posts.
	 *
	 * @return void
	 */
	public static function enqueue_single_assets(): void {

		if ( ! is_singular( 'blog' ) ) {
			return;
		}

		$style_path = BUILD_PATH . 'blocks/blog/article/blog-post.css';

		if ( file_exists( $style_path ) ) {
			wp_enqueue_style(
				'blog-article-style',
				BUILD_URL . 'blocks/blog/article/blog-post.css',
				[],
				(string) filemtime( $style_path )
			);
		}

		$script_path = BUILD_PATH . 'blocks/blog/article/blog-post.asset.php';
		if ( file_exists( $script_path ) ) {
			$asset = include $script_path;
			$deps  = is_array( $asset ) && isset( $asset['dependencies'] ) ? $asset['dependencies'] : [];
			$ver   = is_array( $asset ) && isset( $asset['version'] ) ? $asset['version'] : null;

			wp_enqueue_script(
				'blog-article-script',
				BUILD_URL . 'blocks/blog/article/blog-post.js',
				$deps,
				$ver,
				true
			);
		}
	}

	/**
	 * Filters allowed blocks based on post type and user role
	 *
	 * @param array|bool $allowed_block_types Allowed block types
	 * @param WP_Block_Editor_Context $editor Block editor context
	 *
	 * @return array|bool $allowed_block_types Modified allowed block types
	 */
	public static function allowed_blocks( mixed $allowed_block_types, WP_Block_Editor_Context $editor ): array|bool {

		// Define the allowed block types.
		$allowed_block_types = [
			// 'core/heading',
			// 'core/paragraph',
			// 'core/list',
			// 'core/list-item',
			// 'core/image',
			// 'core/block'
		];

		$post = $editor->post ?? null;

		if ( $post && $post->post_type === 'post' ) {
			return true; // или return $allowed_block_types;
		}

		$allowed_block_types = array_merge( $allowed_block_types, static::get_page_blocks() );

		if ( $post && $post->post_type === 'wp_block' ) {
			$allowed_block_types = array_merge( $allowed_block_types, static::get_pattern_blocks() );
		}

		// Remove duplicates
		return array_unique( $allowed_block_types );
	}

	/**
	 * Returns a list of allowed Gutenberg blocks for patterns
	 *
	 * @return array List of allowed blocks
	 */
	private static function get_pattern_blocks(): array {
		return [
			'acf/some-pattern-block',
		];
	}

	/**
	 * Returns a list of allowed Gutenberg blocks for pages
	 *
	 * @return array List of allowed blocks
	 */
	private static function get_page_blocks(): array {
		return [
			// home
			'acf-block/home-hero',
			'acf-block/slider-standards',
			'acf-block/about-the-company',
			'acf-block/team',
			'acf-block/standards',
			'acf-block/why-us',
			'acf-block/tools',
			'acf-block/our-partners',
			'acf-block/certification-process',
			'acf-block/our-customers',
			'acf-block/contact-from',
			'acf-block/accordion',
			'acf-block/hero-certification',
			'acf-block/about-standard',
			'acf-block/online-course',
			'acf-block/benefits-certification',
			'acf-block/form'
		];
	}


	/**
	 * Returns a list of allowed Gutenberg blocks for posts
	 *
	 * @return array List of allowed blocks
	 */
	private static function get_post_blocks(): array {
		return [
			// 'core/footnotes',
			// 'core/separator',
			// 'core/quote',
		];
	}
}
