<?php

declare( strict_types=1 );

namespace Headless\Modules\Blocks;

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Class BlockDependencies
 *
 * Handles the dependencies for Gutenberg blocks.
 */

class BlockDependencies {
	/**
	 * Add main layout style as a dependency for the editor block styles
	 * register_block_type currently does not support dependencies and versions for block styles
	 * https://github.com/WordPress/WordPress/blob/master/wp-includes/blocks.php#L332
	 *
	 * @return void
	 */
	public static function add_dependencies(): void {

		// Get registered styles
		$styles = wp_styles();

		// Check if the styles object is valid and an array
		if ( isset( $styles->registered ) && is_array( $styles->registered ) ) {

			// Loop through the registered styles
			foreach ( $styles->registered as $handle => $style ) {

				// Check if the handle starts with 'acf-' or 'react-'
				if ( ! str_starts_with( $handle, 'acf-block' )  && ! str_starts_with( $handle, 'react-block' ) ) {
					continue;
				}

				// Add dependencies for the editor
				if ( str_ends_with( $handle, 'editor-style' ) ) {

					// Add the main layout style as a dependency
					$style->deps[] = 'headless-editor-style';

					// Get the block asset data
					if ( ! empty( $style->extra['path'] ) ) {
						$dir        = dirname( $style->extra['path'] );
						$file_name  = $dir . '/block.json';
						$asset_data = static::get_block_asset_data( $file_name );

						if ( $asset_data ) {
							// Set the version
							$style->ver = $asset_data['version'];
						}
					}
					
				}

			}
		}

		// Get registered scripts
		$scripts = wp_scripts();

		// Check if the scripts object is valid and an array
		if ( isset( $scripts->registered ) && is_array( $scripts->registered ) ) {

			// Loop through the registered scripts
			foreach ( $scripts->registered as $handle => $script ) {

				// Check if the handle starts with 'acf-' or 'react-'
				if ( ! str_starts_with( $handle, 'acf-block' )  && ! str_starts_with( $handle, 'react-block' ) ) {
					continue;
				}

				// Add dependencies for the editor
				if ( str_ends_with( $handle, 'editor-script' ) ) {
					$script->deps[] = 'headless-editor-script';
				}
			}
		}
	}

	/**
	 * Gets the asset data for a block, if the asset file exists and contains a version.
	 *
	 * @param string $block_json_path The path to the block.json file.
	 *
	 * @return array|false The asset data if found, null otherwise.
	 */
	private static function get_block_asset_data( string $block_json_path ): ?array {

		// Get the directory containing the block.json file
		$block_dir = dirname( $block_json_path );

		// Try to find the .asset.php file (usually named index.asset.php)
		$asset_file = $block_dir . '/editor.asset.php';

		if ( file_exists( $asset_file ) ) {
			// Load the data from the asset file
			$asset_data = include $asset_file;

			// Check if we got an array and it contains a version
			if ( is_array( $asset_data ) && isset( $asset_data['version'] ) ) {
				return $asset_data;
			}
		}


		// If we didn't find a version, return false
		return false;
	}
}