<?php

declare( strict_types=1 );

namespace Headless\Modules\BlockTermsSync;

use Headless\Modules\BlockTermsSync\Interfaces\BlockTermsSyncHandler;
use Headless\Interfaces\Registrable;
use Headless\Traits\Loadable;
use WP_Post;

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Class BlockTermsSyncManager
 *
 * Handles the synchronization of terms in Gutenberg blocks
 */
class BlockTermsSyncManager implements Registrable {
	use Loadable;

	/**
	 * Handlers for the Gutenberg blocks
	 *
	 * @var array
	 */
	private array $handlers = [];

	/**
	 * Class constructor
	 *
	 * @return void
	 */
	public function __construct() {

		// Add handlers for the Gutenberg blocks
		$this->handlers['acf-block/blog-article'] = new Blocks\BlogArticle();
	}


	/**
	 * Register the hooks for the class
	 *
	 * @return void
	 */
	public function register(): void {
		add_action( 'save_post', [ $this, 'handle_post_save' ], 20, 3 );
	}

	/**
	 * Handle the post save action
	 *
	 * @param int $post_id The post ID
	 * @param WP_Post $post The post object
	 * @param bool $update Whether the post is being updated
	 *
	 * @return void
	 */
	public function handle_post_save( int $post_id, WP_Post $post, bool $update ): void {
		if ( defined( 'DOING_AUTOSAVE' ) && DOING_AUTOSAVE ) {
			return;
		}
		if ( wp_is_post_revision( $post_id ) ) {
			return;
		}

		$blocks = parse_blocks( $post->post_content );
		foreach ( $blocks as $block ) {
			$block_name = $block['blockName'] ?? '';
			if ( $block_name && isset( $this->handlers[ $block_name ] ) && $this->handlers[ $block_name ] instanceof BlockTermsSyncHandler ) {
				$this->handlers[ $block_name ]->sync( $post_id, $block['attrs']['data'] ?? [] );
			}
		}
	}
}

BlockTermsSyncManager::load();