<?php

declare( strict_types=1 );

namespace Headless\Modules\Assets;

use Headless\Interfaces\Registrable;
use Headless\Traits\Loadable;
use Exception;

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Class AssetsManager
 *
 * Manages the enqueueing of assets for the theme, including scripts and styles.
 *
 */

class AssetsManager implements Registrable {

	use Loadable;

	/**
	 * Register the assets by hooking into WordPress
	 *
	 * @return void
	 */
	public function register(): void {
		add_action( 'enqueue_block_assets', 	[ $this, 'enqueue_editor_assets' ] );
		add_action( 'admin_enqueue_scripts', 	[ $this, 'enqueue_admin_assets' ] );
		add_action( 'wp_enqueue_scripts', 		[ $this, 'enqueue_front_assets' ] );
		add_action( 'upload_mimes', 			[ $this, 'my_custom_mime_types' ] );
	}

	/**
	 * Enqueue assets for the editor
	 *
	 * @return void
	 * @throws Exception if asset metadata is not found
	 */
	public function enqueue_editor_assets(): void {
		$this->enqueue_assets( 'editor' );
	}

	/**
	 * Enqueue assets for the admin area
	 *
	 * @return void
	 * @throws Exception if asset metadata is not found
	 */
	public function enqueue_admin_assets(): void {
		$this->enqueue_assets( 'admin' );
	}

	/**
	 * Enqueue assets for the editor
	 *
	 * @return void
	 * @throws Exception if asset metadata is not found
	 */
	public function enqueue_front_assets(): void {
		$this->enqueue_assets( 'main' );
		if ( is_archive() || is_page_template( 'archive.php') ) {
			$this->enqueue_assets( 'blog' );
		}
		if ( is_singular( 'post' ) ) {
			$this->enqueue_assets( 'blog-single' );
		}
	}
	/**
	 * Enqueue assets by entrypoint
	 *
	 * This method adds the specified entrypoint's JavaScript and CSS files to the WordPress queue
	 * It checks for the existence of asset metadata and handles errors appropriately
	 *
	 * @param string $entrypoint The entrypoint to enqueue (e.g., 'editor', 'frontend')
	 *
	 * @return void
	 * @throws Exception if asset metadata is not found
	 */
	private function enqueue_assets( string $entrypoint ): void {

		// Add a trailing slash to the entrypoint if it doesn't already have one
		$slashed_entrypoint = trailingslashit( $entrypoint );

		// Build the base path for the asset files
		$base = $slashed_entrypoint . $entrypoint;

		// Check if the asset file exists
		$metadata = $this->get_asset_metadata( BUILD_PATH . $base . '.asset.php' );
		if ( ! is_array( $metadata ) ) {
			throw new Exception( "AssetsManager: Metadata not found for entrypoint '$entrypoint'" );
		}

		$metadata['dependencies'] = array_merge( ['jquery'], $metadata['dependencies'] ?? [] );

		// Add custom dependencies, we want to enqueue our scripts last
		if ( $entrypoint === 'editor' ) {
			$metadata['dependencies'][] = 'yoast-seo-block-editor';
			$metadata['dependencies'][] = 'yoast-seo-admin-global';
			$metadata['dependencies'][] = 'yoast-seo-post-edit';
			$metadata['dependencies'][] = 'acf-pro-input';
			$metadata['dependencies'][] = 'select2';
		}

		wp_enqueue_script(
			"headless-{$entrypoint}-script",
			BUILD_URL . $base . '.js',
			$metadata['dependencies'],
			$metadata['version']
		);

		wp_enqueue_style(
			"headless-{$entrypoint}-style",
			BUILD_URL . $base . '.css',
			[],
			$metadata['version']
		);

	}

	public function my_custom_mime_types( $mimes ) {
 
		// New allowed mime types.
		$mimes['svg'] = 'image/svg+xml';
		$mimes['svgz'] = 'image/svg+xml';
		$mimes['json'] = 'application/json';
		 
		// Optional. Remove a mime type.
		unset( $mimes['exe'] );
		 
		return $mimes;
	}	  

	/**
	 * Get asset metadata
	 *
	 * @param string $file The asset file path
	 *
	 * @return array|false The asset metadata or false if not found
	 */
	private function get_asset_metadata( string $file ): bool|array {

		// Get the current environment
		$env = defined( 'ENV' ) ? ENV : 'develop';

		// Get the cache key
		$cache_key = 'asset_metadata_' . md5( $file );

		// Check if the cache is valid
		if ( $env === 'production' ) {
			$cached = get_transient( $cache_key );
			if ( $cached !== false ) {
				return $cached;
			}
		}

		// Get the asset file path
		if ( ! file_exists( $file ) ) {
			return false;
		}

		// Include the asset file to get the metadata
		$asset_data = include $file;

		// Check if the asset data is an array and contains the required keys
		if ( ! is_array( $asset_data ) || ! isset( $asset_data['dependencies'], $asset_data['version'] ) || ! is_array( $asset_data['dependencies'] ) ) {
			error_log( "AssetsManager: Invalid metadata structure in $file" );

			return false;
		}

		$result = [
			'dependencies' => $asset_data['dependencies'],
			'version'      => $asset_data['version'],
		];

		if ( $env === 'production' ) {
			set_transient( $cache_key, $result, HOUR_IN_SECONDS );
		}

		return $result;
	}

}

AssetsManager::load();
