<?php

declare( strict_types=1 );

namespace Headless\Kernel;

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Class Dependencies
 *
 * Manages and checks external plugin dependencies required for the theme
 */
class Dependencies {

	/**
	 * List of required plugins with their details
	 *
	 * @var array Array of required plugins
	 */
	private static array $required_plugins = [
		'advanced-custom-fields-pro' => [
			'name'        => 'Advanced Custom Fields Pro',
			'slug'        => 'advanced-custom-fields-pro/acf.php',
			'min_version' => '6.4.0',
			'url'         => 'https://www.advancedcustomfields.com/pro/'
		],
		'wordpress-seo'              => [
			'name'        => 'Yoast SEO',
			'slug'        => 'wordpress-seo/wp-seo.php',
			'min_version' => '24.9',
			'url'         => 'https://ru.wordpress.org/plugins/wordpress-seo/'
		],
	];

	/**
	 * Minimum required PHP version
	 *
	 * @var string Minimum required PHP version
	 */
	private static string $min_php_version = '8.2.0';

	/**
	 * Checks if all required plugins are installed and active
	 *
	 * @return bool True if all dependencies are met, false otherwise
	 */
	public static function check(): bool {
		$all_met = true;

		// Check PHP version
		if ( version_compare( PHP_VERSION, self::$min_php_version, '<' ) ) {
			$all_met = false;
			add_action( 'admin_notices', fn() => self::render_php_version_notice() );
		}

		foreach ( self::$required_plugins as $key => $plugin ) {

			// Check if the plugin is active
			if ( $key === 'advanced-custom-fields-pro' ) {

				// Because ACF Pro is a premium plugin, we check if it's active using a different method
				if ( ! self::is_acf_pro_active() ) {
					$all_met = false;
					add_action( 'admin_notices', fn() => self::render_missing_notice( $plugin ) );
				} elseif ( ! self::check_plugin_version( $plugin['slug'], $plugin['min_version'] ) ) {
					$all_met = false;
					add_action( 'admin_notices', fn() => self::render_version_notice( $plugin ) );
				}
			} else {
				if ( ! self::is_plugin_active( $plugin['slug'] ) ) {
					$all_met = false;
					add_action( 'admin_notices', fn() => self::render_missing_notice( $plugin ) );
				} elseif ( ! self::check_plugin_version( $plugin['slug'], $plugin['min_version'] ) ) {
					$all_met = false;
					add_action( 'admin_notices', fn() => self::render_version_notice( $plugin ) );
				}
			}
		}

		// If any dependencies are not met, show a general error notice
		if ( ! $all_met ) {
			add_action( 'admin_notices', [ self::class, 'render_dependency_error_notice' ] );
		}

		return $all_met;
	}

	/**
	 * Checks if a plugin is active
	 *
	 * @param string $plugin_slug The plugin's main file (e.g., 'wp-graphql/wp-graphql.php')
	 *
	 * @return bool True if the plugin is active, false otherwise
	 */
	private static function is_plugin_active( string $plugin_slug ): bool {
		// Get the list of active plugins as an array
		$active_plugins = (array) get_option( 'active_plugins', [] );

		// Check if the plugin is present in the list of active plugins
		return in_array( $plugin_slug, $active_plugins, true );
	}

	/**
	 * Checks if ACF Pro is active by verifying the presence of a Pro-specific function
	 *
	 * @return bool True if ACF Pro is active, false otherwise
	 */
	private static function is_acf_pro_active(): bool {
		return class_exists( 'ACF' );
	}

	/**
	 * Checks if the plugin version meets the minimum requirement
	 *
	 * @param string $plugin_slug The plugin's main file
	 * @param string $min_version Minimum required version
	 *
	 * @return bool True if the plugin version is greater than or equal to the minimum version, false otherwise
	 */
	private static function check_plugin_version( string $plugin_slug, string $min_version ): bool {
		if ( ! function_exists( 'get_plugin_data' ) ) {
			require_once ABSPATH . 'wp-admin/includes/plugin.php';
		}

		$plugin_data     = get_plugin_data( WP_PLUGIN_DIR . '/' . $plugin_slug, false, false );
		$current_version = $plugin_data['Version'] ?? '0.0.0';

		return version_compare( $current_version, $min_version, '>=' );
	}

	/**
	 * Renders an admin notice for a missing plugin
	 *
	 * @param array $plugin Plugin details
	 *
	 * @return void
	 */
	private static function render_missing_notice( array $plugin ): void {
		echo '<div class="notice notice-error"><p>' . wp_kses_post( "{$plugin['name']} is required. Install it from <a href=\"{$plugin['url']}\">here</a>." ) . '</p></div>';
	}

	/**
	 * Renders an admin notice for an outdated plugin version
	 *
	 * @param array $plugin Plugin details
	 *
	 * @return void
	 */
	private static function render_version_notice( array $plugin ): void {
		echo '<div class="notice notice-error"><p>' . esc_html( "{$plugin['name']} requires version {$plugin['min_version']} or higher." ) . '</p></div>';
	}

	/**
	 * Renders an admin notice for an outdated PHP version
	 *
	 * @return void
	 */
	private static function render_php_version_notice(): void {
		echo '<div class="notice notice-error"><p>Headless theme requires PHP version ' . esc_html( self::$min_php_version ) . ' or higher. Current version: ' . esc_html( PHP_VERSION ) . '.</p></div>';
	}

	/**
	 * Renders a general admin notice if any dependencies are not met
	 *
	 * @return void
	 */
	public static function render_dependency_error_notice(): void {
		echo '<div class="notice notice-error"><p>Headless theme cannot function without required plugins.</p></div>';

	}
}