# Headless WordPress Starter Theme

A modern, production-ready starter theme for headless WordPress development, leveraging GraphQL, ACF, and Tailwind CSS. This theme is designed for developers who want to use WordPress as a backend CMS and build modern, decoupled frontends with maximum flexibility and best practices.

---

## What is Headless WordPress?

**Headless WordPress** means using WordPress only for content management, while the frontend (what users see) is built with modern JavaScript frameworks or static site generators. This approach gives you:
- Full control over the frontend stack (React, Vue, Next.js, etc.)
- Clean separation of concerns
- API-driven development via GraphQL or REST
- Better developer experience and performance

This starter theme keeps the WordPress admin and block editor (Gutenberg) fully functional for content creators, while exposing all data via GraphQL for your frontend app.

---

## Stack Overview

- **WordPress**: The world's most popular CMS, used as a headless backend.
- **GraphQL**: Query any content, fields, blocks, or meta via a powerful API layer (WPGraphQL).
- **Advanced Custom Fields (ACF)**: Flexible content modeling and block creation.
- **Tailwind CSS v3**: Utility-first CSS framework for rapid UI development.
- **Composer**: PSR-4 autoloading, modular PHP architecture, and dependency management.
- **Webpack & @wordpress/scripts**: Modern JS/SCSS build pipeline for blocks and editor assets.

---

## Key Features

### Modular PHP Architecture
- **PSR-4 Autoloading:** All classes in `/app` are autoloaded using Composer, following modern PHP standards.
- **Registrable Interface:** Every module must implement a `register()` method for consistent WordPress integration.
- **Loadable Trait:** Use `Loadable` to quickly register modules with the Core.
- **Centralized Bootstrapping:** All modules are listed in `/app/config.php` and loaded automatically by `/app/bootstrap.php`.

### Custom Gutenberg Blocks
- **Blocks as First-Class Citizens:** Blocks are organized in `/blocks`, each with its own JS, PHP, and SCSS.
- **ACF Integration:** Use ACF to define block fields and render PHP templates.
- **Live Preview:** Editor JS enables dynamic preview and interactivity inside Gutenberg.

### Frontend Workflow
- **Tailwind CSS v3:** Utility classes for rapid, consistent styling. SCSS is also supported for custom styles.
- **Webpack & @wordpress/scripts:** All block/editor JS and SCSS are built and optimized for production.
- **Hot Reloading:** Use `npm run watch` for instant feedback during development.

### Developer Experience
- **Modern Tooling:** Composer, npm, and WP Scripts for a streamlined workflow.
- **Dependency Checks:** The theme checks for required plugins (WPGraphQL, ACF Pro, etc.) and notifies you if anything is missing or outdated.

### GraphQL API Integration
- **WPGraphQL:** Exposes all content, custom fields, and blocks via GraphQL API for decoupled frontends.
- **Custom Types & Fields:** Registers custom GraphQL types and fields for blocks, including ACF fields.
- **Block Data Processing:** Recursively processes and exposes Gutenberg block data (including nested/inner blocks) in the API.

### Asset Management & Caching
- **Dynamic Asset Enqueueing:** Automatically enqueues JS/CSS for editor and frontend, using metadata for dependencies and versioning.
- **Production Caching:** In production, asset metadata and GraphQL block responses are cached using WordPress transients for improved performance.

### Editor UX Enhancements
- **Disable Links in Editor:** Prevents accidental navigation by disabling all links inside the Gutenberg editor.
- **Category/Post Title Sync:** Custom scripts keep category lists and H1 titles in sync with editor state.

---

## Caching Functionality

This theme uses WordPress transients to cache asset metadata and GraphQL block responses in production mode. This reduces database and file system calls, improving performance for both editors and frontend consumers. Caching is automatically invalidated on asset rebuild or content update.

---

## Installation

1. **Upload the theme files** to your WordPress theme directory.
2. **Install PHP dependencies:**
   ```bash
   composer install
   ```
3. **Install JS dependencies:**
   ```bash
   npm install
   ```

---

## NPM Scripts

- `npm run start` – Build all assets and start the file watcher.
- `npm run build` – Build all assets for production.
- `npm run watch` – Start the watcher for development.
- `npm run build:editor` / `npm run build:blocks` – Build only editor or block assets.
- `npm run watch:editor` / `npm run watch:blocks` – Watch only editor or block assets.

---

## Usage Examples

### Registering a New Module
```php
namespace Headless\Modules\Example;

use Headless\Interfaces\Registrable;
use Headless\Traits\Loadable;

class ExampleModule implements Registrable {
    use Loadable;
    public function register() {
        // Register hooks, GraphQL fields, etc.
    }
}

// At the end of the file:
ExampleModule::load();
```
Add `'Headless\Modules\Example\ExampleModule'` to the `modules` array in `app/config.php`.

### Creating a Custom Block
- Place your block in `blocks/your-block/` with `block.json`, `editor.js`, `render.php`, and SCSS as needed.
- Register the block in your PHP module (see BlocksManager).
- Use ACF for custom fields and PHP rendering.

---

## How the Architecture Works

1. **PSR-4 Autoloading:**  
   Composer loads classes from the `/app` directory using the `Headless\\` namespace.
2. **Registrable Interface:**  
   Modules must implement a `register()` method to integrate with WordPress hooks or GraphQL.
3. **Loadable Trait:**  
   Classes using this trait can be registered with a single static call (`ClassName::load()`), which adds them to the core registry.
4. **Central Bootstrapping:**  
   `bootstrap.php` loads all modules defined in `config.php`, ensuring that all necessary functionality is registered automatically.

---

## Project Structure

```
headless/
├── app/                 # PHP modules, traits, interfaces, and core
├── blocks/              # Custom Gutenberg blocks (JS, PHP, SCSS)
├── build/               # Compiled assets
├── resources/           # Editor JS, SCSS, and utility scripts
├── vendor/              # Composer dependencies
├── composer.json        # Composer config and autoload
├── package.json         # NPM scripts and JS dependencies
├── tailwind.config.js   # Tailwind CSS config
├── webpack.config.js    # Webpack config
└── README.md            # Project documentation
```

---

## FAQ

### Why not follow the default WordPress coding standards?
This starter uses PSR-4 and modern PHP conventions for better maintainability, readability, and IDE support. Composer-based autoloading.

### Can I use this theme with any frontend?
Yes! You can build your frontend in React, Vue, Next.js, Nuxt, Svelte, Astro, or any framework that can consume a GraphQL API.

### How do I add a new block/module?
- For PHP modules: Create a new class in `app/Modules`, implement `Registrable`, use `Loadable`, and add it to `config.php`.
- For blocks: Add a new folder to `blocks/`, configure `block.json`, block will be registered automatically.

---

## Contributing

Feel free to open issues or submit pull requests for improvements or bug fixes.
